
__author__    = "Andre Merzky"
__copyright__ = "Copyright 2012-2013, The SAGA Project"
__license__   = "MIT"


import radical.utils.signatures as rus

import saga.adaptors.base       as sab
import saga.attributes          as sa
import saga.session             as ss
import saga.task                as st
import saga.url                 as surl

from   saga.messages.constants  import *
from   saga.constants           import SYNC, ASYNC, TASK


# ------------------------------------------------------------------------------
#
class Endpoint (sa.Attributes) :

    # --------------------------------------------------------------------------
    #
    @rus.takes   ('Endpoint', 
                  rus.optional ((surl.Url, basestring)), 
                  rus.optional (int, rus.nothing), # topology
                  rus.optional (int, rus.nothing), # reliability
                  rus.optional (int, rus.nothing), # atomicity
                  rus.optional (int, rus.nothing), # ordering
                  rus.optional (int, rus.nothing), # correctness
                  rus.optional (sab.Base), 
                  rus.optional (dict), 
                  rus.optional (rus.one_of (SYNC, ASYNC, TASK)))
    @rus.returns (rus.nothing)
    def __init__ (self, url=None, reliability=None, topology=None, 
                  atomicity=None, ordering=None,    correctness=None, 
                  _adaptor=None, _adaptor_state={}, _ttype=None) : 

        # param checks
        if not topology    : topology    = ANY
        if not reliability : reliability = ANY
        if not atomicity   : atomicity   = ANY
        if not ordering    : ordering    = ANY
        if not correctness : correctness = ANY
        url = surl.Url (url)

        self._super = super  (Endpoint, self)
        self._super.__init__ (url, flags, session, 
                                _adaptor, _adaptor_state, _ttype=_ttype)

        # set attribute interface properties
        self._attributes_allow_private (True)
        self._attributes_camelcasing   (True)
        self._attributes_extensible    (False)

        # register properties with the attribute interface 
        self._attributes_register   (URL        , None, sa.STRING, sa.SCALAR, sa.READONLY)
        self._attributes_register   (RECEIVERS  , None, sa.STRING, sa.VECTOR, sa.READONLY)
        self._attributes_register   (STATE      , ANY , sa.ENUM  , sa.SCALAR, sa.READONLY)
        self._attributes_register   (TOPOLOGY   , ANY , sa.ENUM  , sa.SCALAR, sa.READONLY)
        self._attributes_register   (RELIABILITY, ANY , sa.ENUM  , sa.SCALAR, sa.READONLY)
        self._attributes_register   (ATOMICITY  , ANY , sa.ENUM  , sa.SCALAR, sa.READONLY)
        self._attributes_register   (ORDERING   , ANY , sa.ENUM  , sa.SCALAR, sa.READONLY)
        self._attributes_register   (CORRECTNESS, ANY , sa.ENUM  , sa.SCALAR, sa.READONLY)
        
        self._attributes_set_enums  (STATE      , [OPEN, CLOSED     ])
        self._attributes_set_enums  (TOPOLOGY   , [POINT_TO_POINT   , 
                                                   MULTICAST        , 
                                                   PUBLISH_SUBSCRIBE, 
                                                   PEER_TO_PEER     ])
        self._attributes_set_enums  (RELIABILITY, [UNRELIABLE       , 
                                                   CONSISTENT       , 
                                                   SEMI_RELIABLE    , 
                                                   RELIABLE         ])
        self._attributes_set_enums  (ATOMICITY  , [AT_MOST_ONCE     , 
                                                   AT_LEAST_ONCE    , 
                                                   EXACTLY_ONCE     ])
        self._attributes_set_enums  (ORDERING   , [UNVERIFIED       , 
                                                   VERIFIED         ])
        self._attributes_set_enums  (CORRECTNESS, [UNORDERED        , 
                                                   ORDERED          , 
                                                   GLOBALLY_ORDERED ])

        # metrics
        self._attributes_register   (CONNECT    , ANY, sa.STRING, sa.SCALAR, sa.READONLY)
        self._attributes_register   (CLOSE      , ANY, sa.STRING, sa.SCALAR, sa.READONLY)
        self._attributes_register   (MESSAGE    , ANY, sa.STRING, sa.SCALAR, sa.READONLY)


    # --------------------------------------------------------------------------
    #
    @classmethod
    @rus.takes   ('Endpoint', 
                  rus.optional ((surl.Url, basestring)), 
                  rus.optional (int, rus.nothing),  # topology   
                  rus.optional (int, rus.nothing),  # reliability
                  rus.optional (int, rus.nothing),  # atomicity  
                  rus.optional (int, rus.nothing),  # ordering   
                  rus.optional (int, rus.nothing),  # correctness
                  rus.optional (ss.Session), 
                  rus.optional (rus.one_of (SYNC, ASYNC, TASK)))
    @rus.returns (st.Task)
    def create (cls, url=None,  reliability=None, topology=None, 
                atomicity=None, ordering=None,    correctness=None, 
                _ttype=None):

        if not topology    : topology    = ANY
        if not reliability : reliability = ANY
        if not atomicity   : atomicity   = ANY
        if not ordering    : ordering    = ANY
        if not correctness : correctness = ANY

        if not session :
            session = ss.Session (default=True)

        return cls(url, topology, reliability, atomicity, ordering, correctness,
                   session, _ttype=ttype)._init_task


    # --------------------------------------------------------------------------
    #
    # class methods
    #
    @rus.takes   ('Endpoint', 
                  rus.optional (rus.one_of (SYNC, ASYNC, TASK)))
    @rus.returns ((saga.Url, st.Task))
    def get_url (self, ttype=None) :

        '''
        ttype:         saga.task.type enum
        ret:           saga.Url / saga.Task
        
        Return the complete url pointing to the endpoint.

        The call will return the complete url pointing to
        this endpoint as a saga.Url object.
        '''
        return self._adaptor.get_url (ttype=ttype)


    # --------------------------------------------------------------------------
    #
    @rus.takes   ('Endpoint', 
                  rus.optional (rus.one_of (SYNC, ASYNC, TASK)))
    @rus.returns ((rus.list_of(saga.Url), st.Task))
    def get_receivers (self, ttype=None) :

        return self._adaptor.get_receivers (ttype=ttype)


    # --------------------------------------------------------------------------
    #
    @rus.takes   ('Endpoint', 
                  rus.optional (rus.one_of (SYNC, ASYNC, TASK)))
    @rus.returns ((rus.list_of (rus.anything), st.Task))
    def _attribute_lister (self, ttype=None) :

        return self._adaptor.attribute_lister ()


    # --------------------------------------------------------------------------
    #
    @rus.takes   ('Endpoint', 
                  basestring, 
                  int, 
                  callable, 
                  rus.optional (rus.one_of (SYNC, ASYNC, TASK)))
    @rus.returns ((rus.anything, st.Task))
    def _attribute_caller (self, key, id, cb, ttype=None) :

        return self._adaptor.attribute_caller (key, id, cb)



    # ----------------------------------------------------------------
    #
    # advert methods
    #
    @rus.takes   ('Endpoint', 
                  (surl.Url, basestring), 
                  float, 
                  rus.optional (rus.one_of (SYNC, ASYNC, TASK)))
    @rus.returns ((rus.nothing, st.Task))
    def set_ttl  (self, tgt=None, ttl=-1.0, ttype=None) : 
        """
        tgt :           saga.Url / None
        ttl :           int
        ttype:          saga.task.type enum
        ret:            None / saga.Task
        """

        if tgt  :  return self._adaptor.set_ttl      (tgt, ttl, ttype=ttype)
        else    :  return self._adaptor.set_ttl_self (     ttl, ttype=ttype)

     
    # --------------------------------------------------------------------------
    #
    @rus.takes   ('Endpoint', 
                  rus.optional ((surl.Url, basestring)), 
                  rus.optional (rus.one_of (SYNC, ASYNC, TASK)))
    @rus.returns ((float, st.Task))
    def get_ttl  (self, tgt=None, ttype=None) : 
        """
        tgt :           saga.Url / None
        ttype:          saga.task.type enum
        ret:            int / saga.Task
        """

        if tgt  :  return self._adaptor.get_ttl      (tgt, ttype=ttype)
        else    :  return self._adaptor.get_ttl_self (     ttype=ttype)


    # --------------------------------------------------------------------------
    #
    @rus.takes   ('Endpoint', 
                  rus.optional (basestring),
                  rus.optional (basestring),
                  rus.optional ((basestring, object)),
                  rus.optional (int, rus.nothing),
                  rus.optional (rus.one_of (SYNC, ASYNC, TASK)))
    @rus.returns ((rus.list_of (surl.Url), st.Task))
    def find     (self, name_pattern, attr_pattern=None, obj_type=None,
                  flags=RECURSIVE, ttype=None) : 
        """
        name_pattern:   string
        attr_pattern:   string
        obj_type:       string
        flags:          flags enum
        ret:            list [saga.Url]
        """
        
        if not flags : flags = 0
        if attr_pattern or obj_type : 
            return self._adaptor.find_adverts (name_pattern, attr_pattern, obj_type, flags, ttype=ttype)
        else :
            return self._nsdirec.find         (name_pattern,                         flags, ttype=ttype)





